/* -*-C-*-
 ##############################################################################
 #
 # File:        test/order.c
 # RCS:         $Id: sca_ord.c,v 1.4 1998/09/16 17:00:11 dhiller Exp $
 # Description: Host program for doing order tracking with user data
 # Created:     November 10, 1997
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that will demonstrate order tracking on 
 # a single E1432.  It will display time waveforms and resampled time
 # waveforms for all channels.  It will print out the RPM values of both 
 # tach channels at the trigger points.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432 with at least 4 input channels and a tachometer option AYD
 #
 # Set the logical address of the E1432 to the address specified in the LA1 
 # defined below (default 8).
 #
 # Hook up a sweeping signal generator as such:
 #    output set to 2 volts peak sine wave.
 #    tee output to go to the channel 1 input and both tachometer inputs 
 #
 # To run the measurement:
 #    start the program
 #    sweep the frequency between 5Hz(300 RPM) and 105Hz (6300 RPM)
 #    over 30 seconds
 #
 # The program should display the waveform in the channel 1 display on
 # the screen.  The measurement pre-arms at 500 RPM and begins triggering and
 # updating the display at 700 RPM and every 100 RPM interval after that
 # until 6000 rpm. The arming RPM points, RPM at trigger point of the arming
 # tach channel, and average RPM at during the data block of other
 # tach channel are displayed in the terminal window.
 #
 # If the trigger delay is left at -BLOCK_SIZE/2 and the trigger mode
 # is changed to E1432_TACH_EDGE_TRIGGER, the center point of the block
 # of data will remain stationary since we are triggering on a  tach edge
 # and taking data +/- BLOCK_SIZE/2 on either side of that trigger.
 #
 # Note that the RPMs for the two channels are not exactly the same.  The
 # RPM of the arm/trigger channel is calculated using a quadratic fitting 
 # method across tach edge times near the trigger point, while the second
 # channel's RPM is calculated by taking the average RPM for the block.
 #
 # Altering some of the default measurement parameters below will change the
 # nature of the display.  In particular, changing the trigger mode to
 # E1432_AUTO_TRIGGER will produce a display whose center is the exact point
 # at which the RPM reached the arming RPM and won't be lined up with a zero
 # crossing of the tach.  
 #
 # Revisions:
 #
 # Modified from original demo/order.c to exercise user data mode, check
 # framing.
 #
 #
 ##############################################################################
 */
#include <math.h>
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include "e1432.h"
#include "xplot.h"
#include "err1432.h"

/*#define EXT_TRIG */
/*#define TRAILER*/

#define WIDTH           140
#define HEIGHT          80 
#define WARNING_MAX	100

#define RANGE		2.0
#define BLOCK_SIZE	1024
#define	RPM_INTERVAL	100
#define RPM_HIGH	6000
#define	RPM_LOW		700
#define	RPM_PRE_ARM	500
#define SPAN		6400
#define MAX_ORDERS	10.0
#define DELTA_ORDER	1.0
#define PRE_ARM_MODE	E1432_ARM_RPM_RUNUP
#define ARM_MODE	E1432_ARM_RPM_RUNUP 
#define TRIG_MODE	E1432_AUTO_TRIGGER

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int  i, j, nchan;
    SHORTSIZ16 status;
    LONGSIZ32 count;
    SHORTSIZ16 la = 8;
    SHORTSIZ16 chan_list[18];
    SHORTSIZ16 inputs, tachs, arm, nonarm, tach0, tach1;
    E1432ID hw;
    float rpm, rpm2, lastrpm = -1.0;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[16];
    FLOATSIZ32 *resampled[16];
    long points = BLOCK_SIZE;
    char *plotid[32];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange;
    struct e1432_trailer trailer;
    int trig_chan_state;
    SHORTSIZ16 warning[WARNING_MAX]; 
    unsigned long warningCount;
    SHORTSIZ16 meas_state;
    char *semabinloc = "/opt/e1432/lib/sema.bin";

    int do_user_data = 1;
    int print_user_data = 0;
    int frame_check = 1;
    SHORTSIZ16 frame_length = 64;
    SHORTSIZ16 word_length = 32;
    SHORTSIZ16 sub_length = 16;
    SHORTSIZ16 sub_pos = 0;
    LONGSIZ32 user_decimation = 1;
    LONGSIZ32 user_frames_per_block = 1;
    LONGSIZ32 user1_blocksize;
    LONGSIZ32 user1_data[64];

    for(i=0; i<16; i++)
    {
	data[i] = NULL;
	resampled[i] = NULL;
	plotid[i] = NULL;
	plotid[i+16] = NULL;
    }

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
									la);
    (void) fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1432 at la %d ... ",
					   semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    nchan = cf.input_chans;

    (void) printf("Found %d input channels\n", nchan);
    (void) printf("Wait for initial rpm readout before sweeping input\n");

    /* Create channel group */
    for(i=0; i<nchan; i++)
       chan_list[i] = E1432_INPUT_CHAN(i+1);

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n",
			    inputs);
	return -1;
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, inputs, 0x400)); 
#endif

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, RANGE));

    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_32_SERV));
    CHECK(e1432_set_blocksize(hw, inputs, BLOCK_SIZE)); 

    CHECK(e1432_set_data_mode(hw, inputs, E1432_DATA_MODE_OVERLAP_BLOCK));

    if ( do_user_data )
    {
        user1_blocksize = user_frames_per_block * frame_length;
        /* set user data SCA interface parms */
        CHECK(e1432_set_user_data(hw, inputs, frame_length, word_length,
          sub_length, sub_pos));
        CHECK(e1432_set_user_decimation(hw, inputs, user_decimation));
        CHECK(e1432_set_user_frames_per_block(hw, inputs,
	  user_frames_per_block));

        /* turn on USER1 calcs, data */
        CHECK(e1432_set_enable(hw, inputs, E1432_ENABLE_TYPE_USER1,
          E1432_ENABLE_ON));
    }
    else
    {
	print_user_data = 0;
    }

    row = col = 0;
    id = 0;
    temp = (float)(points - 1);

    for(i=0; i < nchan; i++) {
      if(data[i] == NULL) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Can't malloc data array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   data[i][j] = 0.0;
	}
      }

      if(resampled[i] == NULL) {
        resampled[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!resampled[i]) {
          (void) printf("Can't malloc resampled array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   resampled[i][j] = 0.0;
	}
      }

      CHECK(e1432_get_range(hw, chan_list[i], &floatRange));

      if(plotid[id] == NULL) { 
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	(void) sprintf(title, "Time %d", i + 1);

        plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
	(void) sprintf(title, "Resamp %d", i + 1);

        plotid[id+8] = xplot_init_plot(resampled[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
      }
      xplot_change_yautoscale(plotid[id], 0);
      xplot_change_yautoscale(plotid[id+8], 0);
      xplot_set_xscale(plotid[id], (float)0.0, temp);
      xplot_set_xscale(plotid[id+8], (float)0.0, temp);
      xplot_set_yscale(plotid[id], floatRange, -floatRange);
      xplot_set_yscale(plotid[id+8], floatRange, -floatRange);
      xplot_change_xlabel(plotid[id], "Samples");
      xplot_change_xlabel(plotid[id+8], "Samples");
      xplot_change_ylabel(plotid[id], "Volts");
      xplot_change_ylabel(plotid[id+8], "Volts");
      xplot_repaint(plotid[id]); 
      xplot_repaint(plotid[id+8]); 
        
      col++;
      id++;
      if((col % 8) == 0) {
          col = 0;
          row +=2;
          id += 8;
      }

    }

    tach0 = chan_list[nchan] = E1432_TACH_CHAN(1);
    tach1 = chan_list[nchan + 1] = E1432_TACH_CHAN(2);
    tachs = e1432_create_channel_group(hw, 2, &chan_list[nchan]);
    if (tachs >= 0)
    {
	(void) printf("e1432_create_channel_group for tachs returned %d\n",
			    						tachs);
	exit(0);
    }

    /* This sets up the backplane sample clock, SAMPLE0 at 65536 S/s but
       sets the ADC oversample clock, SAMPLE1, such that the ADC sample
       rate is half that, 32768 S/s. */
    CHECK(e1432_set_clock_freq(hw, inputs, 65536));
    CHECK(e1432_set_decimation_undersamp(hw, inputs, 2));

    CHECK(e1432_set_span(hw, inputs, SPAN));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.0));

    arm = tach0;
    nonarm = tach1;

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, inputs, E1432_APPEND_STATUS_ON));
#endif

    CHECK(e1432_set_pre_arm_mode(hw, tachs, PRE_ARM_MODE)); 
    CHECK(e1432_set_arm_mode(hw, tachs, ARM_MODE)); 
    CHECK(e1432_set_auto_trigger(hw, tachs, TRIG_MODE)); 
    CHECK(e1432_set_delta_order(hw, tachs, DELTA_ORDER)); 
    CHECK(e1432_set_max_order(hw, tachs, MAX_ORDERS)); 
    CHECK(e1432_set_rpm_low(hw, tachs, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, tachs, RPM_HIGH)); 
    CHECK(e1432_set_rpm_interval(hw, tachs, RPM_INTERVAL)); 

    CHECK(e1432_set_calc_data(hw, inputs, E1432_DATA_RESAMP_TIME));  
    CHECK(e1432_set_decimation_output(hw, inputs, E1432_MULTIPASS)); 
    CHECK(e1432_set_decimation_oversample(hw, inputs, 
					  E1432_DECIMATION_OVERSAMPLE_ON));

#if	TRIG_MODE == E1432_AUTO_TRIGGER
    trig_chan_state = E1432_CHANNEL_OFF;
#else
    trig_chan_state = E1432_CHANNEL_ON;
#endif

#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, nonarm, trig_chan_state));
    CHECK(e1432_set_trigger_slope(hw, nonarm, E1432_TRIGGER_SLOPE_NEG));
#else
    CHECK(e1432_set_trigger_channel(hw, arm, trig_chan_state));
#endif

    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON));

#if 1
    CHECK(e1432_set_trigger_delay(hw, tachs, -BLOCK_SIZE/2)); 
#else
    CHECK(e1432_set_trigger_delay(hw, tachs, 0)); 
#endif

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));

    do /* wait for meas to start */
    {
        CHECK(e1432_get_current_rpm(hw, arm, &rpm));
    }while(rpm <= 0.0);

    (void) printf("current rpm = %g\n", rpm);

    for (;;)
    {
	do  /* Wait for block available and check for errors and warnings  */
	{
            CHECK(e1432_read_register(hw, arm, 	
					E1432_IRQ_STATUS2_REG, &status));
	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed\n");
		else
		    (void) printf("Fifo overflowed\n");
		exit(0);
	    }

	    if(status & E1432_IRQ_MEAS_WARNING)
	    {
		/* read out all measurement warnings */
		while(status & E1432_IRQ_MEAS_WARNING)
		{
		    CHECK(e1432_get_meas_warning(hw, inputs, warning, 
						WARNING_MAX, &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("%d Warning", warningCount);
			if(warningCount > 1) (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		    CHECK(e1432_read_register(hw, arm, 	
					E1432_IRQ_STATUS2_REG, &status));
		}
	    }

	    CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }

	}while(e1432_block_available(hw, inputs) == 0); 

#ifndef TRAILER
	CHECK(e1432_get_data_rpm(hw, arm, &rpm));
	CHECK(e1432_get_data_rpm(hw, nonarm, &rpm2));
	(void) printf("data rpm = %g    %g\n", rpm, rpm2);

	if(lastrpm != -1.0 && TRIG_MODE == E1432_AUTO_TRIGGER &&
	   ARM_MODE != E1432_AUTO_ARM) 
	{
	    if(fabs((double)rpm - (double)lastrpm) != RPM_INTERVAL)
		(void) printf("********* skipped an rpm interval ********\n");
	}
	lastrpm = rpm;
#endif

#if 0
	CHECK(e1432_get_next_arm_rpm(hw, arm, &rpm));
	(void) printf("next arm rpm = %g\n", rpm);
#endif

	for(i=0; i < nchan; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, chan_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }

#ifdef TRAILER
	    if(i==0) 
		(void) printf("time trailer rpm = %g    %g,  info = 0x%x\n", 
				trailer.rpm1, trailer.rpm2, trailer.info);
#endif

	}

	id = 0;
        for(i=0; i < nchan; i++) {
            if(!plotid[id]) break;
            
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_RESAMP_DATA, resampled[i], points, 
			&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }

#ifdef TRAILER 
	    if(i==0) 
		(void) printf("resamp trailer rpm = %g    %g,  info = 0x%x\n", 
				trailer.rpm1, trailer.rpm2, trailer.info);
#endif

            xplot_check_events(plotid[id]);
            xplot_check_events(plotid[id+8]);
            xplot_data_update(plotid[id]);
            xplot_data_update(plotid[id+8]);    
	    if((++id%8)==0) id += 8;
        }
        if ( do_user_data )
	{
	    for ( i = 0; i < nchan; i++ )  /* read user data */
	    {
	        CHECK(e1432_read_raw_data(hw, chan_list[i], E1432_USER1_DATA,
		  user1_data, user1_blocksize, &trailer, &count));
	        if ( i == 0 && print_user_data )  /* just print 1 channel */
	        {
                    for ( j = 0; j < count; j++ )
		    {
                        printf("  0x%8.8x", user1_data[j]);
                        if ( (j+1) % 4 == 0 ) printf("\n");
	            }
	            printf("\n"); /* separate scans */
	        }
                if ( frame_check )
                {
                    for ( j = 0; j < user_frames_per_block; j++ )
                    {
                        if ( user1_data[j*frame_length]
		          % (frame_length*user_decimation) != 0 )
                        {
                            printf("frame start err, ch %d, frame %d = 0x%x\n",
	                     chan_list[i], j+1, user1_data[j*frame_length]);
                        }
                    }
                }
	    }
	}
    }
    /*NOTREACHED*/
    return 0;
}

